--A collection of functions for SRT editing
--LUA 5.1.2

--[[
	THE BEER-WARE LICENSE (Revision 42):
	<der_ghulbus@ghulbus-inc.de> wrote this file. 
	As long as you retain this notice you can do whatever you want with this stuff. 
	If we meet some day, and you think this stuff is worth it, you can buy me a beer in return. 

	 Andreas Weis
  ]]

--[[ LAYOUT of the SRT-Table:
	local t = {
		{
			id = 1,
			start = {h = 0, m = 0, s = 0, ms = 0},
			fin   = {h = 0, m = 0, s = 0, ms = 0},
			content = { "line1", "line2" }
		},
	}
]]

function PrintTable(t, str)			--helper function: prints complete contents of a table
	if(not str) then str=""; end
	for k,v in pairs(t) do
		local kstr;
		if(type(k) == "string") then kstr = ("\""..k.."\""); else kstr = k; end
		if(type(v) == "table") then
			print(str.."["..kstr.."] = {");
			PrintTable(v, (str.."  "));
			print(str.."},");
		else
			if(type(v) == "number") then
				print(str.."["..kstr.."] = "..v..",");
			elseif(type(v) == "string") then
				print(str.."["..kstr.."] = \""..v.."\",");
			elseif(type(v) == "boolean") then
				if(v) then print(str.."["..k.."] = true,"); else print(str.."["..k.."] = false,"); end
			else
				print(str.."["..kstr.."] = <"..type(v)..">,");
			end
		end
	end
end

function ParseFile(f)				--Parses an srt file into table structure
	local ret = {{}};
	local index = 1;			--index of the currently processed entry
	local state = 0;			--states of the parser: 0-header, 1-timestamp, 2-content
	for line in f:lines() do
		if(state == 0) then		   --id expected
			if(string.find(line, "%D")) then
				print("Parsing error: ID expected: "..line);
				break;
			elseif(line == "") then
				--ignore multiple seperator lines
			else
				ret[index].id = tonumber(line);
				state = 1;
			end
		elseif(state == 1) then		--timestamps expected
			--get timestamps:
			for h,m,s,ms in string.gfind(line, "(%d+):(%d+):(%d+),(%d+)") do
				if(not ret[index].start) then
					ret[index].start = { h=h, m=m, s=s, ms=ms };
				else
					ret[index].fin = { h=h, m=m, s=s, ms=ms };
				end
			end
			state = 2;
		elseif(state == 2) then		--content expected
			if(line == "") then			--seperator reached, proceed with next entry
				index = index + 1;
				ret[index] = {};
				state = 0;
			else					--save content to table
				if(not ret[index].content) then ret[index].content = {}; end
				ret[index]["content"][(#ret[index]["content"])+1] = line;
			end
		else
			print("An error state was reached #"..state);
			return nil;
		end
	end
	if(ret[index].id == nil) then ret[index] = nil; end		--remove last entry if empty
	return ret;
end

function GetTimestampString(t)		--helper function: returns a string representation of the given timestamp table
	for k,v in pairs(t) do
		if(k == "ms") then
			if(type(v) == "number") then
				if(v < 10) then v = ("00"..v); elseif(v < 100) then v = ("0"..tostring(v)); else v = tostring(v); end
			end
		else
			if(type(v) == "number") then
				if(v < 10) then v = ("0"..tostring(v)); else v = tostring(v); end
			end
		end
	end
	return((t.h)..":"..(t.m)..":"..(t.s)..","..(t.ms));
end

function GetTimestampNumber(t)		--helper function: returns the timestamp's value in miliseconds
	return t.ms + t.s*1000 + t.m*60000 + t.h*3600000;
end

function WriteFile(f, t)			--write the table back to an srt file
	for k,v in pairs(t) do
		f:write(v.id); f:write("\n");
		f:write(GetTimestampString(v.start).." --> "..GetTimestampString(v.fin)); f:write("\n");
		for k2, v2 in pairs(v.content) do
			f:write(v2); f:write("\n");
		end
		f:write("\n");
	end
end

function AdjustIndices(t)			--recalculate all ids in an srt table
	local i = 1;
	local done = false;
	for k,v in pairs(t) do
		while(not(t[k]["content"])) do
			print("Found empty entry at #"..(v.id));
			table.remove(t, k);
			v = t[k];
		end
		v.id = i;
		i = i + 1;
	end
end

function AdjustLineCount(t)			--merges three-line entries into two lines if possible
	for k,v in pairs(t) do
		if(#(t[k]["content"]) > 2) then
			--parse lines into word table:
			local words = {};
			local charcount = 0;
			for i=1, #(t[k]["content"]) do
				for w in string.gfind(t[k]["content"][i], "%S+") do table.insert(words, w); charcount = charcount + #w + 1; end
			end
			
			--check for umergable lines:
			local minlinecount = 0;
			for k,v in pairs(words) do
				if((v == "-") and (k ~= 1)) then	--newline required
					minlinecount = minlinecount + 1;
				end
			end
			if(minlinecount > 1) then
				print("!WARNING! Three-line entry at #"..k.." could not be merged: Too many lines.");
			elseif(minlinecount == 1) then		--one minlinecount
				--build new lines:
				local line1, line2 = "", "";
				while(true) do
					if((words[1] == "-") and (#line1 > 1)) then break; end
					local b = (line1.." "..words[1]);
					if(#b > 45) then break; end
					line1 = b; table.remove(words, 1);
				end
				while(true) do
					if(#words == 0) then break; end
					if((words[1] == "-") and (#line2 > 1)) then break; end
					local b = (line2.." "..words[1]);
					if(#b > 45) then break; end
					line2 = b; table.remove(words, 1);
				end
				if(#words > 0) then 
					print("!WARNING! Three-line entry at #"..k.." could not be merged: Unmergable lines.");
				else						
					print("Merging entry #"..k);
					--print(line1);  print(line2);
					t[k]["content"] = {string.sub(line1, 2), string.sub(line2, 2)};
				end
			else							--no minlinecount		
				--build new lines:
				local line1, line2 = "", "";
				while(true) do
					if(#words == 0) then break; end
					local b = (line1.." "..words[1]);
					if((#b > 40) or (#b >= charcount/2)) then break; end
					line1 = b; table.remove(words, 1);
				end
				while(true) do
					if(#words == 0) then break; end
					local b = (line2.." "..words[1]);
					if(#b > 45) then break; end
					line2 = b; table.remove(words, 1);
				end
				if(#words > 0) then 
					print("!WARNING! Three-line entry at #"..k.." could not be merged: Too long.");
				else						
					print("Merging entry #"..k);
					--print(line1);  print(line2);
					t[k]["content"] = {string.sub(line1, 2), string.sub(line2, 2)};
				end
			end
		end	
	end
end

function CheckWordCount(t)			--checks the character limits for each line
	for k,v in pairs(t) do
		for i=1, #(t[k]["content"]) do
			--do not count markup tags!
			if(#(string.gsub( string.gsub(t[k]["content"][i], "<%a>", ""), "</%a>", "") ) > 45) then
				print("!WARNING! Entry #"..k.." exceeds 45 character limit.");
			elseif(#(string.gsub( string.gsub(t[k]["content"][i], "<%a>", ""), "</%a>", "") ) > 40) then
				print("Entry #"..k.." exceeds 40 character limit.");
			end
		end
	end
end

function CheckDurations(t)			--checks if the display duration is sufficient (very strict!)
	local lasttime = 0;
	local totalerrors = 0;
	for k,v in pairs(t) do
		local charcount = 0;
		local wordcount = 0;
		local errors = 0;
		local singleline;
		if(#(t[k]["content"]) == 1) then singleline = true; end
		for i=1, #(t[k]["content"]) do
			for w in string.gfind(t[k]["content"][i], "%S+") do 
				wordcount = wordcount + 1;
				charcount = charcount + #w + 1; 
			end
		end
		local timestart = GetTimestampNumber(v["start"]);
		local timestop  = GetTimestampNumber(v["fin"]);
		if(timestart > timestop) then
			print("!WARNING! Timestamp corrupted at #"..k); errors = errors + 5;
		else
			local delta = timestop - timestart;
			if(delta < 1500) then
				print("Minimum display time undershot at #"..k.." ("..delta.."ms)"); errors = errors + 1;
			end
			if(timestart - lasttime < 250) then
				print("Insufficient timeout before #"..k.." ("..(timestart-lasttime).."ms)"); errors = errors + 1;
			end
			if(singleline and (delta > 3500)) then
				print("Maximum display duration for single line subtitle exceeded at #"..k.." ("..delta.."ms)"); errors = errors + 1;
			elseif(delta > 6000) then
				print("Maximum display duration exceeded at #"..k.." ("..delta.."ms)"); errors = errors + 1;
			end
			if(errors > 2) then print("!WARNING! 2 or more errors for #"..k); end
		end
		totalerrors = totalerrors + errors;
		lasttime = timestop;
	end
	return totalerrors;
end

function AddToTimestamp(t, deltaT)	--helper function: adds deltaT miliseconds to the given timestamp
	local deltaHours = math.floor(deltaT / 3600000);  deltaT = deltaT - deltaHours*3600000;
	local deltaMins  = math.floor(deltaT / 60000);    deltaT = deltaT - deltaMins*60000;
	local deltaSecs  = math.floor(deltaT / 1000);     deltaT = deltaT - deltaSecs*1000;
	local deltaMSecs = deltaT;
	if(deltaHours > 0) then
		t["h"] = tonumber(t["h"]) + deltaHours;
	end
	if(deltaMins > 0) then
		t["m"] = tonumber(t["m"]) + deltaMins;
		if(t["m"] > 59) then
			t["m"] = t["m"] - 60;
			t["h"] = tonumber(t["h"]) + 1;
		end
	end
	if(deltaSecs > 0) then
		t["s"] = tonumber(t["s"]) + deltaSecs;
		if(t["s"] > 59) then
			t["s"] = t["s"] - 60;
			t["m"] = tonumber(t["m"]) + 1;
			if(t["m"] > 59) then
				t["m"] = t["m"] - 60;
				t["h"] = tonumber(t["h"]) + 1;
			end
		end
	end
	if(deltaMSecs > 0) then
		t["ms"] = tonumber(t["ms"]) + deltaMSecs;
		if(t["ms"] > 999) then
			t["ms"] = t["ms"] - 1000;
			t["s"] = tonumber(t["s"]) + 1;
			if(t["s"] > 59) then
				t["s"] = t["s"] - 60;
				t["m"] = tonumber(t["m"]) + 1;
				if(t["m"] > 59) then
					t["m"] = t["m"] - 60;
					t["h"] = tonumber(t["h"]) + 1;
				end
			end
		end
	end
end

function SubFromTimestamp(t, deltaT)		--helper function: subtracts deltaT from the given timestamp
	local deltaHours = math.floor(deltaT / 3600000);  deltaT = deltaT - deltaHours*3600000;
	local deltaMins  = math.floor(deltaT / 60000);    deltaT = deltaT - deltaMins*60000;
	local deltaSecs  = math.floor(deltaT / 1000);     deltaT = deltaT - deltaSecs*1000;
	local deltaMSecs = deltaT;
	if(deltaHours > 0) then
		t["h"] = tonumber(t["h"]) - deltaHours;
		if(t["h"] < 0) then print("FATAL ERROR! ADJUSTED TIMESTAMP TO NEGATIVE!"); return; end
	end
	if(deltaMins > 0) then
		t["m"] = tonumber(t["m"]) - deltaMins;
		if(t["m"] < 0) then
			t["m"] = t["m"] + 60;
			t["h"] = tonumber(t["h"]) - 1;
			if(t["h"] < 0) then print("FATAL ERROR! ADJUSTED TIMESTAMP TO NEGATIVE!"); return; end
		end
	end
	if(deltaSecs > 0) then
		t["s"] = tonumber(t["s"]) - deltaSecs;
		if(t["s"] < 0) then
			t["s"] = t["s"] + 60;
			t["m"] = tonumber(t["m"]) - 1;
			if(t["m"] < 0) then
				t["m"] = t["m"] + 60;
				t["h"] = tonumber(t["h"]) - 1;
				if(t["h"] < 0) then print("FATAL ERROR! ADJUSTED TIMESTAMP TO NEGATIVE!"); return; end
			end
		end
	end
	if(deltaMSecs > 0) then
		t["ms"] = tonumber(t["ms"]) - deltaMSecs;
		if(t["ms"] < 0) then
			t["ms"] = t["ms"] + 1000;
			t["s"] = tonumber(t["s"]) - 1;
			if(t["s"] < 0) then
				t["s"] = t["s"] + 60;
				t["m"] = tonumber(t["m"]) - 1;
				if(t["m"] < 0) then
					t["m"] = t["m"] + 60;
					t["h"] = tonumber(t["h"]) - 1;
					if(t["h"] < 0) then print("FATAL ERROR! ADJUSTED TIMESTAMP TO NEGATIVE!"); return; end
				end
			end
		end
	end
end

--Adjust timestamps between startIndex and endIndex via deltaT miliseconds
function ChangeTiming(t, startIndex, endIndex, deltaT)
	if(not endIndex) then endIndex = #t; end
	print("Adjusting indices #"..startIndex.." to #"..endIndex.." around "..deltaT);
	if(deltaT > 0) then		--move timestamps to later
		for i=startIndex, endIndex do
			AddToTimestamp(t[i]["start"], deltaT);
			AddToTimestamp(t[i]["fin"], deltaT);
		end
	else					--move timestamps to earlier
		deltaT = math.abs(deltaT);
		for i=startIndex, endIndex do
			SubFromTimestamp(t[i]["start"], deltaT);
			SubFromTimestamp(t[i]["fin"], deltaT);
		end
	end
	print("  done.");
end

---------------------------------
--PROGRAM ENTRY
---------------------------------
print("SRT Number Adjuster V-1.0");
print("\nAttempting to open source file: "..filename);
local fin = assert(io.open(filename, "r"));
local data = ParseFile(fin);
--PrintTable(data);
fin:close();
print("Succesfully read "..(#data).." entries.");

print("\nProcessing data...");
AdjustIndices(data);
AdjustLineCount(data);
CheckWordCount(data);
--CheckDurations(data);
--ChangeTiming(data, 1, nil, 400);
print("Done. ("..(#data).." entries left)\n");

print("Saving Backup copy of original file...");
local offset = "";
local t = io.open((filename..".bak"), "r");
while(t) do
	if(type(offset) == "string") then offset = 0; else offset = offset + 1; end
	t = io.open((filename..offset..".bak"), "r");
end
assert(os.rename(filename, (filename..offset..".bak")));

print("Opening output file...");
local fout = assert(io.open(filename, "w"));
print("Writing output to file...");
WriteFile(fout, data);
fout:flush(); fout:close();
print("Done.");