--[[
	THE BEER-WARE LICENSE (Revision 42):
	<der_ghulbus@ghulbus-inc.de> wrote this file. 
	As long as you retain this notice you can do whatever you want with this stuff. 
	If we meet some day, and you think this stuff is worth it, you can buy me a beer in return. 

	 Andreas Weis
  ]]
print("  Loading library for ID3v2 parsing...");
	local filetable = {
		id3_version = nil,			--id3 version number
		id3_revision = nil,			--id3 revision number
		id3_unsynch = nil,			--flag indicating whether unsynchronization for frames is enabled
		id3_extended = nil,			--flag indicating whether an extended header is present 
		id3_footer = nil,			--flag indicating whether a footer is present
		id3_size = nil,				--size of the id3 tag
	};
	-- helper function to convert unsynched integer values:
	function get_unsynch_int(t, startindex, endindex)
		--unsynched ints have the msb of each byte set to zero
		--ignoring the msb of every byte gives the int
		local ret = t[endindex];
		local count = 1;
		for i=endindex-1, startindex, -1 do
			ret = bit.bor(ret, bit.lshift(t[i], count*7));
			count = count + 1;
		end
		return ret;
	end
	
	-- returns a valid frame id string or nil if no valid id was found
	function get_frame_id(t, offs)
		local ret="";
		for i=offs, offs+3 do
			--a valid string may contain uppercase letters and numbers *only*!
			if(not((t[i]>=65 and t[i]<=90) or (t[i]>=48 and t[i]<=57))) then return nil; end
			ret = (ret..string.char(t[i]));
		end
		return ret;
	end
	
	-- parses a single frame and returns a table with entries for id, flags and content
	function parse_frame(b, offs)
		local frame_id = get_frame_id(b, offs);
		if(not(frame_id)) then return nil; end
		--print("    ID: "..frame_id);
		local frame_size = get_unsynch_int(b, offs+4, offs+7);
		--print("    size: "..frame_size);
		local frame_flags = {b[offs+8], b[offs+9]};
		local frame_content = {};
		for i=1, frame_size-1 do
			frame_content[i] = b[offs+10+i];
		end
		return {id=frame_id, size=frame_size, flags=frame_flags, content=frame_content};
	end
	
	-- converts a table holding a char in each cell to a string formed by the chars
	function convert_table2string(t)
		local ret = "";
		for i=1, #t do
			ret = ret..string.char(t[i]);
		end
		return ret;
	end
	-----------------------------------------------------------
	-- parse_header()
	--  IN: b - table with at least 4 entries representing 
	--		the 32 bytes of the MP3 header
	--		fsize - size of the original mp3 file in bytes;
	--		needed for proper computation of frame size
	--  RET: mp3_file table containing various information
	--		about the file but none of the id3 info
	-----------------------------------------------------------
	function parse_header(b, fsize)
		--@todo: extended header support;
		print("Parsing MP3 header...");
		filetable.id3_version = b[4];
		filetable.id3_revision = b[5];
		if(filetable.id3_version > 4) then
			print("ERROR: unknown id3 version");
			return;
		end
		print("  ID3v2 version: "..filetable.id3_version.."."..filetable.id3_revision);
		--parse flags:
		if(bit.band(b[6],0x20) ~= 0) then print("ERROR: experimental tags not supported"); return; end
		filetable.id3_unsynch = bit.rshift(bit.band(b[6],0x80), 7);
		filetable.id3_extended = bit.rshift(bit.band(b[6],0x40), 6);
		filetable.id3_footer = bit.rshift(bit.band(b[6],0x10), 4);
		print("  ID3v2 flags:   unsync:"..filetable.id3_unsynch.."  ext:"..filetable.id3_extended.."  footer:"..filetable.id3_footer);
		if(b[7] ~= 0) then print("ERROR: id3v2 flags seem to be corrupted"); return; end
		--read size:
		filetable.id3_size = get_unsynch_int(b, 8, 11);
		print("  ID3v2 Size:    "..filetable.id3_size.." bytes");
		--skip extended header (if present)
		offset = 0;		--keep this global since it is needed by parse_id3v2()  @todo
		if(filetable.id3_extended == 1) then
			offset = get_unsynch_int(b, 12, 15);
		end
		offset = offset + 11;
		
		print(" done.");
		return {};
	end
	-----------------------------------------------------------
	-- parse_id3v1()
	--  IN: b - table of at least size 128 with the *last* 
	--		128 entries representing the id3v1 tag
	--  RET: id3v1 table
	-----------------------------------------------------------
	function parse_id3v1(b)
		print("@todo");
	end
	-----------------------------------------------------------
	-- parse_id3v2()
	--  IN: b - ??
	--  RET: id3v2 table
	-----------------------------------------------------------
	function parse_id3v2(b)
		--parse id3 frames:
		local ret={};
		local frames = {};
		local i = 1;
		while(true) do			--first parse all present frames to a temporary table
			--print(" Frame #"..i..":");
			frames[i] = parse_frame(b, offset);
			if(not frames[i]) then break; end
			offset = offset + frames[i].size + 10;
			i = i+1;
		end
		print(" Succesfully parsed "..(i-1).." Frames.");
		
		--enumerate parsed frames and extract relevant information:
		for k,v in pairs(frames) do
			if(v.id == "TIT2") then		--title
				ret.Title = convert_table2string(v.content);
			elseif(v.id == "TPE1") then		--artist
				ret.Artist = convert_table2string(v.content);
			elseif(v.id == "TALB") then		--album
				ret.Album = convert_table2string(v.content);
			elseif(v.id == "TYER") then		--year
				ret.Year = convert_table2string(v.content);
			elseif(v.id == "TCON") then		--genre
				ret.Genre = convert_table2string(v.content);
			elseif(v.id == "TRCK") then		--track number
				ret.Track = convert_table2string(v.content);
			elseif(v.id == "COMM") then		--commentary
				ret.Commentary = convert_table2string(v.content);		--@todo this one is still buggy (\0s included)
			elseif(v.id == "TENC") then		--encoded by
				ret.EncodedBy = convert_table2string(v.content);
			elseif(v.id == "TCOP") then		--copyright
				ret.Copyright = convert_table2string(v.content);
			elseif(v.id == "TOPE") then		--original artist
				ret.OriginalArtist = convert_table2string(v.content);
			elseif(v.id == "TCOM") then		--composer
				ret.Composer = convert_table2string(v.content);
			elseif(v.id == "WXXX") then		--url
				ret.URL = convert_table2string(v.content);
			end
		end
		for k,v in pairs(ret) do
			print("  "..k..": "..v);
		end
		return ret;
	end
print("   done.");