--[[
	THE BEER-WARE LICENSE (Revision 42):
	<der_ghulbus@ghulbus-inc.de> wrote this file. 
	As long as you retain this notice you can do whatever you want with this stuff. 
	If we meet some day, and you think this stuff is worth it, you can buy me a beer in return. 

	 Andreas Weis
  ]]
print("  Loading library for ID3v1 parsing...");

	-- the following table holds the parsed file header data before further processing
	local filetable = {
		sync = nil,				-- 11 bits, has to be all 1s
		version = nil,			-- 2 bits,  1=mpeg1.0, 0=mpeg2.0
		layer = nil,			-- 2 bits,  4-lay = layerI, II or III
		error_protec = nil,		-- 1 bit,   0=yes, 1=no
		bitrate_index = nil,	-- 4 bits,  see details below
		sampling_freq = nil,	-- 2 bits,  see details below
		padding = nil,			-- 1 bits
		extension = nil,		-- 1 bits,  see details below
		mode = nil,				-- 2 bits,  see details below
		mode_ext = nil,			-- 2 bits,  used with "joint stereo" mode
		copyright = nil,		-- 1 bits,  0=no 1=yes
		original = nil,			-- 1 bits,  0=no 1=yes
		emphasis = nil,			-- 2 bits,  see details below
	};

	--possible bitrate indices (14 possibilities each)
	local bitrate = {
		mpeg1 = {
					layer1 = {32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448},
					layer2 = {32, 48, 56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320, 384},
					layer3 = {32, 40, 48,  56,  64,  80,  96, 112, 128, 160, 192, 224, 256, 320},
				},
		mpeg2 = {
					layer1 = {32, 48, 56,  64,  80,  96, 112, 128, 144, 160, 176, 192, 224, 256},
					layer2 = { 8, 16, 24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160},
					layer3 = { 8, 16, 24,  32,  40,  48,  56,  64,  80,  96, 112, 128, 144, 160},
				},
	};
	--possible sampling frequencies
	local frequency = {
		mpeg1  = { [0]=44100, [1]=48000, [2]=32000 },
		mpeg2  = { [0]=22050, [1]=24000, [2]=16000 },
		mpeg25 = { [0]=11025, [1]=12000, [2]=8000 },
	};

	local id3_genre = {
		"Classic Rock", "Country", "Dance", "Disco", "Funk", "Grunge", "Hip-Hop", "Jazz", "Metal", "New Age",
		"Oldies", "Other", "Pop", "R&B", "Rap", "Reggae", "Rock", "Techno", "Industrial", "Alternative",
		"Ska", "Death Metal", "Pranks", "Soundtrack", "Euro-Techno", "Ambient", "Trip-Hop", "Vocal", "Jazz+Funk", "Fusion",
		"Trance", "Classical", "Instrumental", "Acid", "House", "Game", "Sound Clip", "Gospel", "Noise", "AlternRock",
		"Bass", "Soul", "Punk", "Space", "Meditative", "Instrumental Pop", "Instrumental Rock", "Ethnic", "Gothic", "Darkwave",
		"Techno-Industrial", "Electronic", "Pop-Folk", "Eurodance", "Dream", "Southern Rock", "Comedy", "Cult", "Gangsta", "Top 40",
		"Christian Rap", "Pop/Funk", "Jungle", "Native American", "Cabaret", "New Wave", "Psychadelic", "Rave", "Showtunes", "Trailer",
		"Lo-Fi", "Tribal", "Acid Punk", "Acid Jazz", "Polka", "Retro", "Musical", "Rock & Roll", "Hard Rock", 
		--beyond this point are the winamp extensions which are *not* part of the standard! comment them out if you don't want them
		"Folk", "Folk-Rock", "National Folk", "Swing", "Fast Fusion", "Bebob", "Latin", "Revival", "Celtic", "Bluegrass",
		"Avantgarde", "Gothic Rock", "Progressive Rock", "Psychedelic Rock", "Symphonic Rock", "Slow Rock", "Big Band", "Chorus", "Easy Listening", "Acoustic",
		"Humour", "Speech", "Chanson", "Opera", "Chamber Music", "Sonata", "Symphony", "Booty Bass", "Primus", "Porn Groove",
		"Satire", "Slow Jam", "Club", "Tango", "Samba", "Folklore", "Ballad", "Power Ballad", "Rhythmic Soul", "Freestyle",
		"Duet", "Punk Rock", "Drum Solo", "A capella", "Euro-House", "Dance Hall", "Goa", "Drum & Bass", "Club-House", "Hardcore",
		"Terror", "Indie", "BritPop", "Negerpunk", "Polsk Punk", "Beat", "Christian Gansta Rap", "Heavy Metal", "Black Metal", "Crossover",
		"Contemporary Christian", "Christian Rock", "Merengue", "Salsa", "Thrash Metal", "Anime", "JPop", "Synthpop",
		--end of non-standard stuff
		"Unknown", [0] = "Blues",
	};

	--helper function: returns the actual length of a null- or space-padded string:
	function get_padded_str_len(str)
		--although the standard requires padding with 0x00, winamp among others is padding with spaces 0x20
		local last_unpadded = 0;
		for i=0, string.len(str) do
			if( (string.byte(str, i) ~= 0x00) and (string.byte(str, i) ~= 0x20) ) then
				last_unpadded = i;
			end
		end
		return last_unpadded;
	end

	-----------------------------------------------------------
	-- parse_header()
	--  IN: b - table with at least 4 entries representing 
	--		the 32 bytes of the MP3 header
	--		fsize - size of the original mp3 file in bytes;
	--		needed for proper computation of frame size
	--  RET: mp3_file table containing various information
	--		about the file but none of the id3 info
	-----------------------------------------------------------
	function parse_header(b, fsize)
		print("Parsing MP3 header...");
		if(not(b[1] == 255 and bit.band(b[2],0xE0) == 0xE0)) then print("ERROR: invalid file"); return; end
		filetable.sync          = 0x7FF
		filetable.version       = bit.rshift(bit.band(b[2],0x18), 3);
		filetable.layer         = bit.rshift(bit.band(b[2],0x06), 1);
		filetable.error_protec  = bit.band(b[2],0x01);
		filetable.bitrate_index = bit.rshift(bit.band(b[3],0xF0), 4);
		filetable.sampling_freq = bit.rshift(bit.band(b[3],0x0C), 2);
		filetable.padding       = bit.rshift(bit.band(b[3],0x04), 1);
		filetable.extension     = bit.band(b[3],0x01);
		filetable.mode          = bit.rshift(bit.band(b[4],0xC0), 6);
		filetable.mode_ext      = bit.rshift(bit.band(b[4],0x30), 4);
		filetable.copyright     = bit.rshift(bit.band(b[4],0x08), 3);
		filetable.original      = bit.rshift(bit.band(b[4],0x04), 2);
		filetable.emphasis      = bit.band(b[4],0x02);
		--for k,v in pairs(filetable) do print(" ["..k.."] - "..v); end

		--Gather file information:
		local bitr;
		local freq;
		if(filetable.version == 3) then
			bitr = bitrate.mpeg1
			freq = frequency.mpeg1;
		elseif(filetable.version == 2) then
			bitr = bitrate.mpeg2
			freq = frequency.mpeg2;
		elseif(filetable.version == 0) then
			--bitr = bitrate.mpeg25;
			freq = frequency.mpeg25;
			print("ERROR: mpeg version 2.5 not supported :-(");
			return;
		else
			print("ERROR: mpeg version tag corrupted - "..filetable.version);
			return;
		end
		if(filetable.layer == 3) then
			bitr = bitr.layer1;
		elseif(filetable.layer == 2) then
			bitr = bitr.layer2;
		elseif(filetable.layer == 1) then
			bitr = bitr.layer3;
		else
			print("ERROR: layer version tag corrupted - "..filetable.layer);
			return;
		end
		if(filetable.bitrate_index > 14) then
			print("ERROR: bitrate index corrupted - "..filetable.bitrate_index);
			return;
		end
		bitr = bitr[filetable.bitrate_index] or 0;
		if(filetable.sampling_freq > 2) then
			print("ERROR: frequency index corrupted - "..filetable.bitrate_index);
			return;
		end
		freq = freq[filetable.sampling_freq];
		print("  Bitrate:   "..bitr.."kbit  (0 means variable bitrate)");
		print("  Frequency: "..freq.."Hz");
		local mode;
		local mode_ext = "";
		if(filetable.mode == 0) then
			mode = "stereo";
		elseif(filetable.mode == 1) then
			mode = "joint stereo";
			if(filetable.mode_ext == 0) then mode_ext = "";
			elseif(filetable.mode_ext == 1) then mode_ext = "(Intensity Stereo)";
			elseif(filetable.mode_ext == 2) then mode_ext = "(MS Stereo)";
			elseif(filetable.mode_ext == 3) then mode_ext = "(Intensity Stereo and MS Stereo)";
			else assert(false); return; end
		elseif(filetable.mode == 2) then
			mode = "dual channel";
		elseif(filetable.mode == 3) then
			mode = "single channel";
		else
			assert(false);
			return;
		end
		print("  Mode:      "..mode..mode_ext);
		local copyr;
		local orig;
		local emph;
		local crcc
		if(filetable.copyright == 0) then copyr = "no"; else copyr = "yes"; end
		if(filetable.original == 0) then orig = "no"; else orig = "yes"; end
		if(filetable.error_protec == 0) then crcc = "no" else crcc = "yes"; end
		if(filetable.emphasis == 0) then emph = "none";
		elseif(filetable.emphasis == 1) then emph = "50/15 ms";
		elseif(filetable.emphasis == 2) then print("ERROR: emphasis bit corrupted - "..filetable.emphasis); return;
		elseif(filetable.emphasis == 3) then emph = "CCIT J.17";
		else assert(false); return; 
		end
		print("  CRC: "..crcc.."  Copyrighted: "..copyr.."  Original: "..orig.."  Emphasis: "..emph);
		local frame_size = math.floor(144000 * bitr / freq + filetable.padding);
		print("  Frame Size: "..frame_size.." bytes  ("..math.floor(fsize/frame_size).." Frames total)");
		
		local mp3_file = {};
		mp3_file.bitr       = bitr;
		mp3_file.freq       = freq;
		mp3_file.mode       = mode;
		mp3_file.mode_ext   = mode_ext;
		mp3_file.copyr      = copyr;
		mp3_file.orig       = orig;
		mp3_file.emph       = emph;
		mp3_file.crcc       = crcc;
		mp3_file.frame_size = frame_size;
		mp3_file.n_frames   = math.floor(fsize/frame_size);
		
		print(" done.");
		return mp3_file;
	end

	-----------------------------------------------------------
	-- parse_id3v1()
	--  IN: b - table of at least size 128 with the *last* 
	--		128 entries representing the id3v1 tag
	--  RET: id3v1 table
	-----------------------------------------------------------
	function parse_id3v1(b)
		print("Parsing ID3v1 tag...");
		--the id3v1 is exactly 128 bytes long and located at the very end of the file
		local tag = {};
		local id3v1 = {};
		for i=#b-127, #b do
			table.insert(tag, b[i]);
		end
		--first 3 bytes hold the word TAG (0-2)
		if(not ((tag[1] == 0x54) and (tag[2] == 0x41) and (tag[3] == 0x47)) ) then
			print("ERROR: no valid id3v1 tag present");
			return;
		end
		--title tag (3-32):
		local tmp="";
		for i=4, 33 do
			tmp = tmp..string.char(tag[i]);
		end
		id3v1.Title = string.sub(tmp, 1, get_padded_str_len(tmp));
		--artist tag (33-62):
		tmp="";
		for i=34, 63 do
			tmp = tmp..string.char(tag[i]);
		end
		id3v1.Artist = string.sub(tmp, 1, get_padded_str_len(tmp));
		--album tag (63-92):
		tmp="";
		for i=64, 92 do
			tmp = tmp..string.char(tag[i]);
		end
		id3v1.Album = string.sub(tmp, 1, get_padded_str_len(tmp));
		--year tag (93-96):
		tmp="";
		for i=94, 97 do
			tmp = tmp..string.char(tag[i]);
		end
		id3v1.Year = string.sub(tmp, 1, get_padded_str_len(tmp));
		--comment tag (97-126):
		tmp="";
		for i=98, 127 do
			tmp = tmp..string.char(tag[i]);
		end
		id3v1.Comment = string.sub(tmp, 1, get_padded_str_len(tmp));
		--genre tag (127):
		if(tag[128] > #id3_genre) then tag[128] = #id3_genre; end
		id3v1.Genre = id3_genre[tag[128]];

		for k,v in pairs(id3v1) do
			print("  "..k..": "..v);
		end
		print(" done.");
		
		return id3v1;
	end
	
	-----------------------------------------------------------
	-- parse_id3v2()
	-- since this library processes id3v1 files, the next one is just a dummy
	-----------------------------------------------------------
	function parse_id3v2(b)
		print("No ID3v2 tag present.");
		return nil;
	end

print("   done.");