--[[
 Converts Logs from Trillian 1.0 Pro to Miranda Message Export Plugin 3.1.0.2 format
 
  November 2007 (contact: der_ghulbus@ghulbus-inc.de)
]]

--[[
	THE BEER-WARE LICENSE (Revision 42):
	<der_ghulbus@ghulbus-inc.de> wrote this file. 
	As long as you retain this notice you can do whatever you want with this stuff. 
	If we meet some day, and you think this stuff is worth it, you can buy me a beer in return. 

	 Andreas Weis
  ]]

--complete path of the logfile
local filename = "C:\\Trillian\\users\\ComicSansMS\\logs\\ICQ\\102086057.log";	
--set this flag if you want to ignore unclosed sessions when encountering a Session Start
--while parsing; this is generally a good idea, since unclosed sessions were a common side
--effect when trillian crashed
local IGNORE_UNCLOSED_SESSIONS_ON_BEGIN = true;

function PrintTable(t, str)			--helper function: prints complete contents of a table
	if(not str) then str=""; end
	for k,v in pairs(t) do
		local kstr;
		if(type(k) == "string") then kstr = ("\""..k.."\""); else kstr = k; end
		if(type(v) == "table") then
			print(str.."["..kstr.."] = {");
			PrintTable(v, (str.."  "));
			print(str.."},");
		else
			if(type(v) == "number") then
				print(str.."["..kstr.."] = "..v..",");
			elseif(type(v) == "string") then
				print(str.."["..kstr.."] = \""..v.."\",");
			elseif(type(v) == "boolean") then
				if(v) then print(str.."["..k.."] = true,"); else print(str.."["..k.."] = false,"); end
			else
				print(str.."["..kstr.."] = <"..type(v)..">,");
			end
		end
	end
end

--read the contents of f into a table, where each 
--table entry corresponds to a line in the file
function ReadFile(f)
	local ret = {};
	for line in f:lines() do
		table.insert(ret, line);
	end
	return ret;	
end

--helper function: converts the three character month-strings of the trillian log
--into number strings as used by the miranda logs
function GetMonthFromString(str)
	local ret;
	if(str == "Jan") then
		ret = "01";
	elseif(str == "Feb") then
		ret = "02";
	elseif(str == "Mar") then
		ret = "03";
	elseif(str == "Apr") then
		ret = "04";
	elseif(str == "May") then
		ret = "05";
	elseif(str == "Jun") then
		ret = "06";
	elseif(str == "Jul") then
		ret = "07";
	elseif(str == "Aug") then
		ret = "08";
	elseif(str == "Sep") then
		ret = "09";
	elseif(str == "Oct") then
		ret = "10";
	elseif(str == "Nov") then
		ret = "11";
	elseif(str == "Dec") then
		ret = "12";
	else
		assert(false, "Unknown month string: "..str);
	end
	return ret;
end

--helper function: called whenever beginning a new line in the miranda log
--used for postprocessing and debugging
function StartNewLine(t)
	if(t["nextline"] == 0) then t["nextline"] = 1;  return; end
	assert(t[t["nextline"]], "Empty line at "..t["nextline"]);
	
	--postprocessing:
	--replace umlauts with utf8
	--  \195\164   \195\182  \195\188  \195\132  \195\150  \195\156  \195\159
	t[t["nextline"]] = string.gsub(t[t["nextline"]], "%", string.char(195, 164));
	t[t["nextline"]] = string.gsub(t[t["nextline"]], "%", string.char(195, 182));
	t[t["nextline"]] = string.gsub(t[t["nextline"]], "%", string.char(195, 188));
	t[t["nextline"]] = string.gsub(t[t["nextline"]], "%", string.char(195, 132));
	t[t["nextline"]] = string.gsub(t[t["nextline"]], "%", string.char(195, 150));
	t[t["nextline"]] = string.gsub(t[t["nextline"]], "%", string.char(195, 156));
	t[t["nextline"]] = string.gsub(t[t["nextline"]], "%", string.char(195, 159));
	
	t["nextline"] = t["nextline"] + 1;
end

--main function; parses trillian log data and returns a table with miranda log data
function ProcessData(t)
	local ret = {};
	ret["nextline"] = 0;	--internal counter
	
	local block_date = nil;
	local block_time = nil;
	
	local scan_str_sessionstart = "Session Start %(ICQ %- (%d+)%:(%C+)%)%: (%w%w%w) (%w%w%w) (%d%d) (%d%d)%:(%d%d)%:(%d%d) (%d%d%d%d)";
	local scan_str_sessionclose = "Session Close %((%C+)%)%: (%w%w%w) (%w%w%w) (%d%d) (%d%d)%:(%d%d):(%d%d) (%d%d%d%d)";
	local scan_str_log = "([^%c%:]+)%:(%C+)";
	local scan_str_logtimestamped = "%[(%d%d)%:(%d%d)%] ([^%c%:]+)%:(%C+)";
	
	local tmp = {};
	--the main loop iterates through all lines
	for i=1, #t do
		local line_done = false;
		if( (string.sub(t[i], 1, 3) == "***") or ((string.sub(t[i], 1, 1) == "[") and (string.sub(t[i], 9, 11) == "***")) ) then
			--system message, ignore
			line_done = true;
		end
		if(not line_done) then
			for uin, nick, wday, month_str, day, hour, min, sec, year in string.gfind(t[i], scan_str_sessionstart) do
				--session start
				assert(IGNORE_UNCLOSED_SESSIONS_ON_BEGIN or (block_date == nil), "Unclosed session at "..i);
				local month = GetMonthFromString(month_str);
				block_date = (day.."."..month.."."..year);
				block_time = (hour..":"..min..":"..sec);
				--print("Session start: "..month_str.." "..day);
				if(not ret["user"]) then
					ret["user"] = nick;
					ret["nick"] = nick;
				end
				line_done = true;
			end
		end
		if(not line_done) then
			for nick, wday, month_str, day, hour, min, sec, year in string.gfind(t[i], scan_str_sessionclose) do
				--session close
				assert(block_date ~= nil, "Unopened session at "..i);
				block_date = nil;
				block_time = nil;
				--print("Session close: "..month_str.." "..day);
				line_done = true;
			end
		end
		if(not line_done) then
			for hour, min, nick, message in string.gfind(t[i], scan_str_logtimestamped) do
				--message with timestamp
				assert(block_date ~= nil, "Unblocked message at "..i);
				StartNewLine(ret);
				ret[ret["nextline"]] = (nick.."  "..block_date.." "..hour..":"..min..":".."00 "..message);
				line_done = true;
			end
		end
		if(not line_done) then
			for nick, message in string.gfind(t[i], scan_str_log) do
				--message without timestamp
				assert(block_date ~= nil, "Unblocked message at "..i);
				StartNewLine(ret);
				ret[ret["nextline"]] = (nick.."  "..block_date.." "..block_time.." "..message);
				line_done = true;
			end
		end
		if(not line_done) then
			if(block_date) then
				--multi line message followup
				ret[ret["nextline"]] = (ret[ret["nextline"]].." "..t[i]);
				line_done = true;
			end			
		end
	end	--for lines	
	return ret;
end

--writes the output file; constructs the header and writes miranda log from t
function WriteFile(f, t)
	f:write(string.char(239, 187, 191));		--UTF8 prefix (EFBBBF)
	--protocol file header:
	f:write("------------------------------------------------\n");
	f:write("      History for\n");
	f:write("User      : "..t["user"].."\n");
	f:write("Protocol  : ICQ\n");
	f:write("UIN       : "..t["uin"].."\n");
	f:write("FirstName : \n");
	f:write("LastName  : \n");
	f:write("Age       : \n");
	f:write("Gender    : \n");
	f:write("e-mail    : \n");
	f:write("Nick      : "..t["nick"].."\n");
	f:write("City      : \n");
	f:write("State     : \n");
	f:write("Phone     : \n");
	f:write("Homepage  : \n");
	f:write("- About -\n");
	f:write("\n");
	f:write("------------------------------------------------\n");
	--protocol data:
	for i=1, t["nextline"] do
		f:write(t[i].."\n");
	end		
end

-------------------------
-- Program Entry       --
-------------------------
print("Converting...");
print("Log Converter V-1.0");
print("\nAttempting to open source file: "..filename);
local fin = assert(io.open(filename, "r"));
print(" done.\n");
print("Reading data...");
local data = ReadFile(fin);
print(" done.\n");
fin:close();

print("Processing data -");
data = ProcessData(data);
assert(data["user"], "No data in file");
data["uin"] = string.gfind(filename "%\\(%d+)%.log")();
print(" done. - "..(data["nextline"]-1).." entries converted.");
--PrintTable(data);

print("\nWriting output...");
local fout = assert(io.open(string.gsub(filename, "%.log", "%.txt"), "w"));
WriteFile(fout, data);
fout:flush(); fout:close();
print(" done.");
