/* File system operations library for Lua 5.x */
/* Weis Andreas <www.ghulbus-inc.de>, Dec 2007 */

/** INSTALLATION INSTRUCTIONS: 
 * In Luas linit.c add the following declaration:
 *    LUALIB_API int luaopen_filesys (lua_State*);
 * right below the #includes and add the following entry to luaL_Reg lualibs[] before {NULL, NULL}:
 *    {"filesys", luaopen_filesys},
 * Recompile, have fun :)
 */

/** Flag for excluding dangerous functions
 * If you do not want the functions for deleting and renaming files and directories
 * to be included for security reasons, uncomment the following #define.
 */
/* #define LUA__FILESYS__EXCLUDE_DANGEROUS_FUNCTIONS */

#ifdef WIN32
#	define _CRT_SECURE_NO_WARNINGS
#	define WIN32_LEAN_AND_MEAN
#	include <windows.h>
#	include <direct.h>
#else
#	include <dirent.h>
#	include <sys/stat.h>
#	include <sys/types.h>
#	include <sys/param.h>
#	include <errno.h>
#	include <unistd.h>
#	include <time.h>
#endif
#include <stdlib.h>
#include <string.h>

#include "lauxlib.h"
#include "lua.h"

/** Lists a file or directory
 */
static int lfilesys_ls(lua_State* L);
/** Prints the current working directory
 */
static int lfilesys_pwd(lua_State* L);
/** Creates a new directory
 */
static int lfilesys_mkdir(lua_State *L);
/** Deletes an empty directory
 */
static int lfilesys_rmdir(lua_State *L);
/** Deletes a file
 */
static int lfilesys_rmfile(lua_State *L);
/** Changes a file's name
 */
static int lfilesys_rename(lua_State *L);


/** Library registration struct
 */
static const struct luaL_reg lfilesyslib[] = {
	{"ls",     lfilesys_ls},
	{"pwd",    lfilesys_pwd},
	{"mkdir",  lfilesys_mkdir},
#ifndef LUA__FILESYS__EXCLUDE_DANGEROUS_FUNCTIONS
	{"rmdir",  lfilesys_rmdir},
	{"rmfile", lfilesys_rmfile},
	{"rm",     lfilesys_rmfile},
	{"rename", lfilesys_rename},
	{"mv",     lfilesys_rename},
#endif
	{NULL, NULL}
};

/** Library Loader
 */
LUALIB_API int luaopen_filesys(lua_State* L) {
	luaL_openlib(L, "filesys", lfilesyslib, 0);
	return 1;
}

#ifdef WIN32
	/**
	 * @section Implementation WIN32
	 * @{
	 */

	/**	Helper function: builds file table from searchstring
	 */
	static int GetFileDataFromName(lua_State* L, const char* fname)
	{
		WIN32_FIND_DATAA fdata;
		HANDLE hf;
		
		/* process search query: */
		hf = FindFirstFileA(fname, &fdata);
		if(hf == INVALID_HANDLE_VALUE) { 
			lua_pushnil(L);
		} else {
			/* write all matches to a table: */
			lua_newtable(L);
			do {
				lua_pushstring(L, fdata.cFileName);
					lua_newtable(L);
					lua_pushstring(L, "directory");
					lua_pushboolean(L, (fdata.dwFileAttributes&0x10));
					lua_settable(L, -3);
				lua_settable(L, -3);
			} while(FindNextFileA(hf, &fdata) == TRUE);

			FindClose(hf);
		}
	
		return 1;
	}

	static int lfilesys_ls(lua_State* L) {
		/* get argument from stack: */
		const char* path_str = luaL_checkstring(L, 1);

		/* build searchstring for FindFirstFile(): */
		size_t slen = strlen(path_str);
		char* buffer = (char*)malloc(slen+3);
		strcpy(buffer, path_str);
		if( (slen == 0) || ((path_str[slen-1] != '/') && (path_str[slen-1] != '\\')) ) {
			strcat(buffer, "/*");
		} else {
			strcat(buffer, "*");
		}
		
		GetFileDataFromName(L, buffer);

		free(buffer);

		return 1;
	}

	static int lfilesys_mkdir(lua_State *L) {
		const char* path_str = luaL_checkstring(L, 1);
		if(_mkdir(path_str) == -1) {
			luaL_error(L, strerror(errno));
		}
		return 0;
	}

	/** @} */
#else
	/**
	 *	@section Implementation GNU Linux
	 *  @{
	 */
	static int lfilesys_ls(lua_State* L) {
		DIR* dp;
		struct dirent* dirp;
		struct stat fdata;
		size_t slen;
		char* buffer;
		/* get argument from stack: */
		const char* path_str = luaL_checkstring(L, 1);

		/* lstat needs the full path, which will reside in this string buffer */
		slen = strlen(path_str) + 1;
		buffer = (char*)malloc(slen+257);

		dp = opendir(path_str);
		if(dp == NULL) {
			lua_pushnil(L); 
		} else {
			/* build table */
			lua_newtable(L);
			while((dirp = readdir(dp)) != NULL) {
				/* save full path to buffer: */
				strcpy(buffer, path_str);
				strcat(buffer, "/");
				strcat(buffer, dirp->d_name);
				/* get file stats: */
				lstat(buffer, &fdata);
				/* write lua table: */
				lua_pushstring(L, dirp->d_name);
					lua_newtable(L);
					lua_pushstring(L, "directory");
					lua_pushboolean(L, S_ISDIR(fdata.st_mode));
					lua_settable(L, -3);
				lua_settable(L, -3);
			}
			closedir(dp);
		}
		free(buffer);

		return 1;
	}

	static int lfilesys_mkdir(lua_State *L) {
		const char* path_str = luaL_checkstring(L, 1);
		if(mkdir(path_str, S_IRWXU) == -1) {
			luaL_error(L, strerror(errno));
		}
		return 0;
	}
	/** @} */
#endif

/**
 *	@section Implementation (platform independent)
 *  @{
 */
static int lfilesys_pwd(lua_State* L) {
	char* buffer = (char*)_getcwd(NULL, 0);
	if(buffer) {
		lua_pushstring(L, buffer);
		free(buffer);
	} else {
		lua_pushnil(L);
	}
	return 1;
}

static int lfilesys_rmdir(lua_State *L) {
	const char* path_str = luaL_checkstring(L, 1);
	if(_rmdir(path_str) == -1) {
		luaL_error(L, strerror(errno));
	}
	return 0;
}

static int lfilesys_rmfile(lua_State *L) {
	const char* path_str = luaL_checkstring(L, 1);
	if(remove(path_str) == -1) {
		luaL_error(L, strerror(errno));
	}
	return 0;
}

static int lfilesys_rename(lua_State *L) {
	const char* path_str = luaL_checkstring(L, 1);
	const char* new_path = luaL_checkstring(L, 2);
	if(rename(path_str, new_path) == -1) {
		luaL_error(L, strerror(errno));
	}
	return 0;
}
/** @} */
